
**********************************************************
*   Example for Event History Data Handling
*       Josef Brderl, March 2011
*
**********************************************************

version 11
cd "K:\Vorlesung EHA\Stata Beispiele\"    //working directory


***********************************************
* List data
***********************************************
use "EH data.dta", clear
list id - educ, sepby(id)


***********************************************
* (marriage) Episode data set
* Dates transformed to process time
***********************************************
use "EH data.dta", clear

* Transform to long format (episode data set)
reshape long ts tf end, i(id) j(episode)
list id episode ts tf end birthy inty educ, sepby(id)

drop if  ts==.                //selecting the valid marriage episodes
replace tf=inty if(tf==.)     //missing tf set to interview year
gen tfp = tf-ts               //tf transformed to process time (=duration)
gen fail = end==1             //failure indicator (1=divorce, 0=censoring)

stset tfp, failure(fail==1)   //declaring the data to be "survival time" 

list id episode tfp fail _t0 _t _d educ, sepby(id)



***********************************************
* (marriage) Episode data set
* Dates left in calendar time
***********************************************
use "EH data.dta", clear
reshape long ts tf end, i(id) j(episode)
drop if  ts==.                //selecting the valid marriage episodes
replace tf=inty if(tf==.)     //missing tf set to interview year
gen fail = end==1             //failure indicator (1=divorce, 0=censoring)

stset tf, origin(time ts) time0(ts) failure(fail==1) 

list id episode ts tf fail _t0 _t _d educ, sepby(id)



**************************************************
* Discrete-time EHA
**************************************************
use "EH data.dta", clear
reshape long ts tf end, i(id) j(episode)
drop if  ts==.             //selecting the valid marriage episodes
replace tf=inty if(tf==.)  //missing tf set to interview year
gen tfp = tf-ts            //tf transformed to process time (=duration)
gen fail = end==1          //failure indicator (1=divorce, 0=censoring)
gen recid = _n             //create an id for each episode (needed for stsplit)
stset tfp, failure(fail==1) id(recid)  
list id recid ts tf fail _t0 _t _d, sepby(id)

* Prepare the data for discrete-time analysis
stsplit T0, every(1)            //"person-period" episode splitting

list id recid ts tf fail _t0 _t _d educ, sepby(episode)



***************************************************************************
*       Generating a panel data structure from Event History data
*
* We produce data that record each year the state
* These data could be used for panel data analysis
* or sequence analysis
***************************************************************************

use "EH data.dta", clear
list id birthy ts1 tf1 end1 ts2 tf2 end2 inty, sepby(id)

* This program assumes that there are two episodes in the dataset (k=1/2). 
* We will loop from age 17 to age 30 and record the states occupied
* 0=single, 1=married, 2=divorced, 3=widowed

gen  ageint  = inty - birthy       //age at interview
egen nepi    = rownonmiss(ts*)     //number of valid marriage episodes

* Preset state 0 (single) over the whole sequence
forvalues j = 17/30  {    
   generate s`j' = 0 if `j' <= ageint            
}
   
if nepi>0 {                   //The rest is only for those who married at least once

  forvalues k=1/2 {
    replace tf`k'  = inty  if `k'==nepi & tf`k'==.    //imputing inty for censored episodes
    replace end`k' = 3     if `k'==nepi & end`k'==.   //flaging censored episodes with end==3
  }
  
  forvalues k=1/2 {                                   //converting years to age 
    replace ts`k' = ts`k' - birthy 
    replace tf`k' = tf`k' - birthy 
  }

  forvalues k=1/2 {                           //setting the endpoint of the sequence
	replace ts`k'=ageint+1 if `k'==nepi+1
  }
  gen  ts3 = ageint+1 if nepi==2
  
  forvalues j=17/30 {
    forvalues k=1/2 {
	  local l=`k'+1
	  quietly replace s`j' = 1  if `j'>=ts`k' & `j'<=tf`k'               //being in marriage
	  quietly replace s`j' = 2  if `j'>=tf`k' & `j'< ts`l' & end`k'==1   //being divorced (last marriage year overwritten)
	  quietly replace s`j' = 3  if `j'>=tf`k' & `j'< ts`l' & end`k'==2   //being widowed (last marriage year overwritten)
    }
  }
}


list id birthy ts1 tf1 end1 ts2 tf2 end2 ts3 inty, sepby(id)

list id s17-s30, sepby(id)


